package gov.va.genisis2.dao.impl;

import javax.persistence.NoResultException;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import gov.va.genisis2.common.enums.CommonEnum;
import gov.va.genisis2.dao.IUserRoleTypeDao;
import gov.va.genisis2.exceptions.ErrorEnum;
import gov.va.genisis2.exceptions.GenisisDAOException;
import gov.va.genisis2.model.UserRoleType;

/**
 *
 * The UserRoleType data access object (DAO) is an object that provides an
 * abstract interface to some type of database or other persistence mechanism.
 * By mapping application calls to the persistence layer, UserRoleType DAO
 * provide some specific data operations without exposing details of the
 * database.
 * 
 * @author Prasad Thummalapalli
 *
 */
@Repository
@Transactional(value = "transactionManager")
public class UserRoleTypeDao extends AbstactHibernateDao implements IUserRoleTypeDao {

	private static final Logger LOGGER = LoggerFactory.getLogger(UserRoleTypeDao.class);

	@Autowired
	private SessionFactory sessionFactory;

	@Override
	public UserRoleType saveOrUpdateUserRoleType(UserRoleType userRoleType) throws GenisisDAOException {
		if (userRoleType == null) {
			throw new GenisisDAOException("userRoleType entity is null");
		}

		UserRoleType persistentUserRoleType;
		try {
			Session session = sessionFactory.getCurrentSession();
			if (userRoleType.getId() <= 0) {
				userRoleType.setId(getMaxRowValue(session, UserRoleType.class, CommonEnum.STATUS_ID.getText()));
			}
			persistentUserRoleType = (UserRoleType) updateData(session, userRoleType);
		} catch (Exception ex) {
			LOGGER.error("Exception occured on createUserRoleType.", ex);
			throw new GenisisDAOException("Exception occured on createUserRoleType.", ex);
		}

		return persistentUserRoleType;
	}

	@Override
	public UserRoleType getUserRoleTypeByUserId(int userId) throws GenisisDAOException {
		UserRoleType userRoleType = null;

		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<UserRoleType> query = criteriaBuilder.createQuery(UserRoleType.class);
			Root<UserRoleType> root = query.from(UserRoleType.class);
			Predicate condition = criteriaBuilder.equal(root.get(CommonEnum.USER_ID.getText()), userId);
			query.where(condition);
			userRoleType = session.createQuery(query).getSingleResult();
		} catch (NoResultException e) {
			// Swallow exception
			LOGGER.error(ErrorEnum.USERROLETYPE_DAO_EXP_GETUSERROLETYPE.getErrorMessage(), userId, e);
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.USERROLETYPE_DAO_EXP_GETUSERROLETYPE.getErrorMessage(), userId, ex);
			throw new GenisisDAOException(ex.getClass().getSimpleName(), ex);
		}

		return userRoleType;
	}

	/**
	 * @param sessionFactory
	 *            the sessionFactory to set
	 */
	public void setSessionFactory(SessionFactory sessionFactory) {
		this.sessionFactory = sessionFactory;
	}
}
